<?php
/* --------------------------------------------------------------
   ParcelServiceReader.php 2020-02-28
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\ParcelService\Repository;

use Doctrine\DBAL\Connection;
use Gambio\Admin\ParcelService\Exceptions\ParcelServiceNotFoundException;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceId;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceSqlCriteria;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceSqlPagination;

/**
 * Class ParcelServiceReader
 *
 * @package Gambio\Admin\ParcelService\Repository
 */
class ParcelServiceReader
{
    /**
     * @var Connection
     */
    private $db;
    
    
    /**
     * ParcelServiceReader constructor.
     *
     * @param Connection $db
     */
    public function __construct(Connection $db)
    {
        $this->db = $db;
    }
    
    
    /**
     * Returns all parcel services data.
     *
     * @param ParcelServiceSqlCriteria   $criteria
     * @param ParcelServiceSqlPagination $pagination
     *
     * @return array
     */
    public function getAllData(
        ParcelServiceSqlCriteria $criteria,
        ParcelServiceSqlPagination $pagination
    ): array {
        $query = $this->db->createQueryBuilder()
            ->select('parcel_services.*')
            ->from('parcel_services')
            ->join('parcel_services',
                   'parcel_services_description',
                   'parcel_services_description',
                   'parcel_services.parcel_service_id = parcel_services_description.parcel_service_id')
            ->groupBy('parcel_services.parcel_service_id');
        $criteria->applyToQuery($query);
        $pagination->applyToQuery($query);
        
        return array_map(function (array $parcelServiceData) {
            return $this->addDescriptionData($parcelServiceData);
        },
            $query->execute()->fetchAll());
    }
    
    
    /**
     * Returns the total number of parcel services.
     *
     * @param ParcelServiceSqlCriteria $criteria
     *
     * @return int
     */
    public function getTotalCount(ParcelServiceSqlCriteria $criteria): int
    {
        $query = $this->db->createQueryBuilder()
            ->select('parcel_services.parcel_service_id')
            ->from('parcel_services')
            ->join('parcel_services',
                   'parcel_services_description',
                   'parcel_services_description',
                   'parcel_services.parcel_service_id = parcel_services_description.parcel_service_id')
            ->groupBy('parcel_services.parcel_service_id');
        $criteria->applyToQuery($query);
        
        return $query->execute()->rowCount();
    }
    
    
    /**
     * Returns the parcel service data.
     *
     * @param ParcelServiceId $id
     *
     * @return array
     *
     * @throws ParcelServiceNotFoundException
     */
    public function getById(ParcelServiceId $id): array
    {
        $parcelServiceData = $this->db->createQueryBuilder()
            ->select('*')
            ->from('parcel_services')
            ->where('parcel_service_id = :parcelServiceId')
            ->setParameter('parcelServiceId', $id->value())
            ->execute()
            ->fetch();
        
        if ($parcelServiceData === false) {
            throw ParcelServiceNotFoundException::forId($id->value());
        }
        
        return $this->addDescriptionData($parcelServiceData);
    }
    
    
    /**
     * @param array $parcelServiceData
     *
     * @return array
     */
    public function addDescriptionData(array $parcelServiceData): array
    {
        $query = $this->db->createQueryBuilder()
            ->select('*')
            ->from('parcel_services_description')
            ->where('parcel_service_id = :parcelServiceId')
            ->setParameter('parcelServiceId', $parcelServiceData['parcel_service_id']);
        
        $parcelServiceData['descriptions'] = $query->execute()->fetchAll();
        
        return $parcelServiceData;
    }
}